#ifndef TWISTER_HEADER_H
#define TWISTER_HEADER_H

#include <iostream.h>

// Provides an argument for annulus::twist()
enum Sign
{
  plus = 1,
  minus = -1
};

// Provides an argument for annulus::twohandle()
enum Direction
{
  above = true,
  below = false
};

// Tells the tetrahedron where in the square or cusp it lives.
enum Position
{
  top, mid, low, up, down, other
};

// Permutations (low-level; used by other classes)
class perm
{
  int image[4];

public:

  perm();

  perm(const perm &to_copy);

  // Image of 0-3 under the permutation
  perm(const int* of_in);

  // Constructor:  perm(0,1,2,3) is the identity
  perm(int im0, int im1, int im2, int im3)
  {
    image[0] = im0;
    image[1] = im1;
    image[2] = im2;
    image[3] = im3;
  }

  // Inverse
  perm inverse() const;

  // Overloaded subscript operator
  int operator[](int input) const;

  // Composition of permutations
  perm of(const perm &other) const;

  // Allows printing
  friend ostream &operator<<(ostream &o, const perm &to_print);
};

// identity permutation
const perm perm0123(0, 1, 2, 3);

// and transpositions
const perm perm1023(1, 0, 2, 3);
const perm perm2103(2, 1, 0, 3);
const perm perm3120(3, 1, 2, 0);
const perm perm0213(0, 2, 1, 3);
const perm perm0321(0, 3, 2, 1);
const perm perm0132(0, 1, 3, 2);

// Tetrahedron  (low-level; used by other classes)
class tetra
{
  tetra *prev;

  tetra *gluedto[4];
  perm gluing[4];

  int index;

  int cusp; // Which cusp vertex 3 lies in.  Usually -1.
  Position position; 
  bool meridian; // true iff position is up/down and 
		 // the tet wins in round two of bundle_capoff().

  void glue(tetra *whereglue, int whichface, const perm &how);

public:

  tetra(class manifold *M, Position mypos);

  int number()
  {
    return index;
  }

  void print_wrt(ostream &o, int lastnumber);

  tetra *lookback()
  {
    return prev;
  }

  int whatcusp()
  {
    return cusp;
  }

  void setcusp(int newcusp)
  {
    cusp = newcusp;
  }

  void you_lose()
  {
    meridian = false;
  }

  tetra *thruface(int whichface) const;

  void grow(manifold *M);
  void neighborglue();
  void walk_about(int fromface);

  void bundle_grow(manifold *M);
  void ups_to_downs();
  int bundle_neighborglue(int fromface);

  // gluesym may only be used to glue a pair of free faces.
  // 'how' is, in our applications, always odd --- i.e. our
  // manifolds are oriented, with all tetrahedra right-handed. 
  void gluesym(tetra *whereglue, int whichface, const perm &how);

  // whichface must be glued already to some face. 
  // 'how' should be an even permutation.
  // See the function body for usage.
  void subbedby(tetra *whereglue, int whichface, const perm &how);
};

// Oriented square
// Provides an input format for annulus and rectangle constructors
struct orisquare
{
  class square *sq;
  bool is_upright;
  orisquare *next;
  int depth;

  orisquare(Sign whichway, class square *mysquare)
  {
    sq = mysquare;
    is_upright = (whichway == plus);
    next = NULL;
    depth = 1;
  }

  orisquare &operator>>(orisquare &other)
  {
    orisquare *iter = this;

    while (iter->next)
    {
      iter->depth += other.depth;
      iter = iter->next;
    }
    iter->depth += other.depth;

    iter->next = &other;

    return *this;
  }
};

// An intersection point, thickened in both directions
// For examples of use, see the file sample.c
class square 
{
  tetra *topleft, *midleft, *lowleft;
  tetra *topright, *midright, *lowright;

  class manifold *home;

  // has it been glued in each direction?
  bool is_glued[2]; 

public:

  // A square needs to know what manifold it belongs to.
  square(class manifold &home_in);

  // Glue the top to the bottom--used when making 
  // surface bundles (or "Mark's manifolds").
  // (Those are built from a pair of compression bodies by gluing 
  // positive boundaries to positive and negative to negative.)
  void closeup();

  // The constructor for annuli and rectangles requires arguments
  // of the form +square, -square.
  orisquare &operator+();
  orisquare &operator-();

  friend class annulus;
  friend class rectangle;
};

// A thickened closed curve, i.e. an annulus.
// For examples of use, see the file sample.c
class annulus
{
  int length;
  square **sq;
  bool *upright;  // true for each + intersection

  void glue_squares();

public:

  // The old way of constructing an annulus: give it an
  // array of pointers to squares, and an array of booleans
  // indicating orientation (true = '+').
  annulus(int length_in, square **sq_in, bool *upright_in);

  // The better way: annulus(2, (orisquare *) {-foo_sq, +bar_sq} )
  // annulus(int length_in, orisquare *osq);

  // Now I like annulus(-foo_sq >> +bar_sq >> -etc_sq);
  annulus(orisquare &ori);

  // If overloaded unary +/- make you queasy, you can also
  // invoke it as   annulus(2, (oriquare *)
  //     {orisquare(plus, &foo_sq), orisquare(minus, &bar_square)} )

  ~annulus();

  // twohandle(above), twohandle(below)
  annulus &twohandle(Direction is_above);

  // For backwards compatibility
  annulus &twohandleabove()
  {
    return twohandle(above);
  }

  annulus &twohandlebelow()
  {
    return twohandle(below);
  }

  // Plumbing on a curve on a surface in a manifold,
  // or cutting regluing after a Dehn twist
  // (positive/negative "speed bump" glued above)
  void twist(Sign whichway);

  // alternate calling method for twist()
  annulus &operator++();  
  annulus &operator--();  

};

// A thickened arc
// See the file sample.c for an example of use
class rectangle 
{
  int length;
  square **sq;
  bool *upright;  // true for each + intersection

  void glue_squares();

public:

  // Same construction options as for annulus
  rectangle(int length_in, square **sq_in, bool *upright_in);
  rectangle(orisquare &ori);
  ~rectangle();
};

// A three-manifold, including a list of its tetrahedra
class manifold
{
  tetra *last;

  char *name;
  int cusp_count;  // set by bundle_capoff()

  tetra *onemore(tetra *newguy);

public:

  manifold(char *name_in);
  ~manifold();

  int size()
  {
    return last? (last->number() + 1) : 0;
  }  

  void capoff();
  void bundle_capoff();

  void snap_print(ostream &o);

  friend tetra::tetra(class manifold *M, Position mypos);
};

#endif
