#include <stdlib.h>
#include <iostream.h>

#include "twister.h"

bool debugging = false;

///////////// Permutation code //////////////////

perm::perm()
{
}

perm::perm(const perm &to_copy)
{
  for (int i=0; i<4; i++)
  {
    image[i] = to_copy.image[i];
  }
}

perm::perm(const int* image_in)
{
  for (int i=0; i<4; i++)
  {
    image[i] = image_in[i];
  }
}

int perm::operator[](int input) const
{
  return image[input];
}

perm perm::inverse() const
{
  perm outp;

  for (int i=0; i<4; i++)
  {
    outp.image[image[i]] = i;
  }

  return outp;
}

// Composition of permutations
perm perm::of(const perm &other) const
{
  perm outp;

  for (int i=0; i<4; i++)
  {
    outp.image[i] = image[other.image[i]];
  }

  return outp;
}

///////////// Tetrahedron code //////////////////

tetra::tetra(manifold *M, Position mypos)
{
  if(debugging)
    { 
      cerr << "tet " << flush;
    }
  for (int i=0; i<4; i++)
  {
    gluedto[i] = NULL;
  }

  prev = M->onemore(this);

  index = prev? (prev->index + 1) : 0;

  position = mypos;
  cusp = -1;
  meridian = false;
}

// The function glue() is sufficiently primitive
// that we will leave all error checking to the calling functions.
void tetra::glue(tetra *whereglue, int whichface, const perm &how)
{
  gluedto[whichface] = whereglue;
  gluing[whichface] = how;
}

// The function gluesym() glues together two tetrahedra
// along faces which it checks are free.
// 
// For oriented manifolds, the permutation "how" should be odd.
//
void tetra::gluesym(tetra *whereglue,
                    int whichface,
                    const perm &how)
{
  if ( gluedto[whichface]  // != NULL
       ||  whereglue->gluedto[how[whichface]])  // != NULL )
  {
    cerr << "Attempt to reglue.  Possible causes:" << endl;
    cerr << "  * self-intersecting curves" << endl;
    cerr << "  * intersecting 2-handles" << endl;
    cerr << "  * 2-handles attached along closed-up squares" << endl;
    cerr << "  * closing up squares or attaching 2-handles after capoff()" << endl;
    exit(1);
  }

  glue(whereglue, whichface, how);
  whereglue->glue(this, how[whichface], how.inverse() );
}

// The function subbedby() breaks open a gluing and replaces
// one of the tetrahedra (A below, glued to B above) by another 
// (D below, now glued to B above like A used to be).
// 
// For oriented manifolds, the permutation "how" should be even.
// 
// It's okay if some of the tetrahedra A, B, and D coincide, but
// the three faces involved must be distinct.
//
void tetra::subbedby(tetra *whereglue, int whichface, const perm &how)
{
  // before:    A-B   D free
  // after:     A free   D-B

  // A: this
  // D: whereglue
  // B: *gluedto[whichface]

  // "how" maps vertices of A to vertices of D
  // (Identity means glue D to B just like A was glued)
  // "how" should be an even permutation.

  perm AtoB(gluing[whichface]);

  if ( gluedto[whichface] == NULL )
  {
    cerr << "Illegal free face encountered.  Did you remember";
    cerr << " to capoff() before twisting?" << endl;
    exit(1);
  }
  if ( whereglue->gluedto[how[whichface]])
  {
    cerr << "Attempt to reglue.  Possible causes:" << endl;
    cerr << "  * self-intersecting curves" << endl;
    cerr << "  * intersecting 2-handles" << endl;
    cerr << "  * 2-handles attached along closed-up squares" << endl;
    cerr << "  * closing up squares or attaching 2-handles after capoff()" << endl;
  }

  // set D's pointers to point to B
  whereglue->glue(gluedto[whichface],
                  how[whichface],
                  AtoB.of(how.inverse()) );

  // set B's pointers to point to D
  gluedto[whichface]->glue(whereglue,
                           AtoB[whichface],
                           how.of(AtoB.inverse()) );

  // free up whichface of A
  gluedto[whichface] = NULL;
  // Ack.  Shouldn't we also empty out the correct entry of A's gluing
  // array?
}

tetra *tetra::thruface(int whichface) const
{
  if (!gluedto[whichface])
  {
    cerr << "Illegal free face encountered.  Did you remember";
    cerr << " to capoff() before twisting?" << endl;
    exit(1);
  }

  return gluedto[whichface];
}

//
// First step in the function capoff().
//
// Glue another tetrahedron onto each
// open face of the current tetrahedron.
void tetra::grow(manifold *M)
{
  if (gluedto[0] == NULL)
  {
    tetra *newguy = new tetra(M,other);
    //    We have no idea why we used to construct a special-purpose permutation here.
    //    newguy->gluesym(this, 3, perm(2, 3, 1, 0));
    newguy->gluesym(this, 3, perm3120);
  }

  if (gluedto[1] == NULL)
  {
    tetra *newguy = new tetra(M,other);
    newguy->gluesym(this, 3, perm0321);
  }

  if (gluedto[2] == NULL)
  {
    tetra *newguy = new tetra(M,other);
    newguy->gluesym(this, 3, perm0132);
  }

  if (gluedto[3] == NULL)
  {
    tetra *newguy = new tetra(M,other);
    newguy->gluesym(this, 3, perm0213);
  }
}

//
// First step in the function bundle_capoff().
//
// Glue another tetrahedron onto each
// open face of the current tetrahedron.
void tetra::bundle_grow(manifold *M)
{
  tetra *newguy;
  Position newguypos;
  if (gluedto[0] == NULL)
  {
    switch(position)
    {
      case top:
        newguypos = up;
        break;
      case mid:
        newguypos = down;
        break;
      cerr << "Illegal free face found by bundle_grow()" << endl;
      exit(1);
    }
    newguy = new tetra(M,newguypos);
    //    We have no idea why we used to construct a special-purpose permutation here.
    //    newguy->gluesym(this, 3, perm(2, 3, 1, 0));
    newguy->gluesym(this, 3, perm3120);
  }

  if (gluedto[1] == NULL)
  {
    switch(position)
    {
      case top:
        newguypos = up;
        break;
      case low:
        newguypos = down;
        break;
      cerr << "Illegal free face found by bundle_grow()" << endl;
      exit(1);
    }
    newguy = new tetra(M,newguypos);
    newguy->gluesym(this, 3, perm0321);
  }

  if (gluedto[2] == NULL)
  {
    switch(position)
    {
      case top:
      case mid:
        newguypos = up;
        break;
      case low:
        newguypos = down;
        break;
      cerr << "Illegal free face found by bundle_grow()" << endl;
      exit(1);
    }
    newguy = new tetra(M,newguypos);
    newguy->gluesym(this, 3, perm0132);
  }

  if (gluedto[3] == NULL)
  {
    cerr << "Illegal free face found by bundle_grow()" << endl;
    exit(1);
  }
  newguy->meridian = true;
}

// Given a non-3 fromface, walk_about the edge j-k
// where {0, 1, 2} = {fromface, j, k}.
// And then glue.
void tetra::walk_about(int fromface)
{
  if(debugging)
    { 
      cerr << "walk " << flush;
    }
  if (gluedto[fromface])
    return;

  // Here's how we start.
  tetra *currenttetra = this;
  tetra *nexttetra;
  perm how, init_how;

  switch (fromface)
  {
  case 2:
    init_how = perm0132;
    break;
  case 1:
    init_how = perm0321;
    break;
  case 0:
    init_how = perm3120;
  } 

  how = init_how;
  
  // Now walk around the edge.
  while( (currenttetra->gluedto[how[fromface]]) != NULL)
  {
    nexttetra = currenttetra->gluedto[how[fromface]];
    how = (currenttetra->gluing[how[fromface]]).of(how.of(init_how));
    currenttetra = nexttetra;
  }

  // We've found the last tetra next to the edge
  // so glue it to the first one.
  gluesym(currenttetra, fromface, how);
}

//
// Second step in the function capoff().
//
// The tetrahedra created by grow() have three
// free faces. These get glued to other tetras,
// also created by grow(), via neighborglue(). 
// It is desperately wrong to use this function 
// for a more general manifold-with-boundary. 
// (eg we don't check that the first tetra is 
// different from the last tetra, etc.)
void tetra::neighborglue()
{
  for (int fromface=0; fromface<3; fromface++)
  {
    if (gluedto[fromface])
      continue;

    walk_about(fromface);
  }
}

///////////// Square code //////////////////

square::square(manifold &home_in)
{
  home = &home_in;

  is_glued[0] = is_glued[1] = false;

  topleft = new tetra(home, top);
  midleft = new tetra(home, mid);
  lowleft = new tetra(home, low);

  topright = new tetra(home, top);
  midright = new tetra(home, mid);
  lowright = new tetra(home, low);

  // We'll glue these together to give
  // two triangular prisms, and then a cube.

  // Gluings inside left prism.
  topleft->gluesym(midleft, 3, perm0132);
  midleft->gluesym(lowleft, 3, perm3120);

  // Gluings inside right prism.
  topright->gluesym(midright, 3, perm0321);
  midright->gluesym(lowright, 3, perm3120);

  // Gluings between the two prisms.
  midleft->gluesym(topright, 1, perm1023);
  lowleft->gluesym(midright, 1, perm1023);
}

void square::closeup()
{
  // Closing up the cube to get a solid torus. 
  // Glue the bottoms to the tops.

  lowleft->gluesym(topleft, 3, perm0321);
  lowright->gluesym(topright, 3, perm0132);
}
  
orisquare &square::operator+()
{
  orisquare *ori = new orisquare(plus, this);

  if (is_glued[1])
  {
    cerr << "Error: a square can only appear once with '+' orientation." << endl;
    exit(1);
  }

  is_glued[1] = true;

  return *ori;
}

orisquare &square::operator-()
{
  orisquare *ori = new orisquare(minus, this);

  if (is_glued[0])
  {
    cerr << "Error: a square can only appear once with '-' orientation." << endl;
    exit(1);
  }

  is_glued[0] = true;

  return *ori;
}

///////////// Annulus code //////////////////

annulus::annulus(orisquare &ori)
{
  length = ori.depth;

  orisquare *iter=&ori, *next;

  // Used to have: sq = new (square*)[length]; 
  // Change suggested by ND 2008/6/23, 
  // to make Twister compile under gcc4.

  sq = new square*[length];
  upright = new bool[length];

  for (int i=0; i<length; i++)
  {
    sq[i] = iter->sq;
    upright[i] = iter->is_upright;
    next = iter->next;
    delete iter;
    iter = next;
  }

  glue_squares();
}

annulus::annulus(int length_in, square **sq_in, bool *upright_in)
{
  length = length_in;

  sq = new square*[length];
  upright = new bool[length];

  for (int i=0; i<length; i++)
  {
    sq[i] = sq_in[i];
    upright[i] = upright_in[i];
  }

  glue_squares();
}

annulus::~annulus()
{
  delete[] sq;
  delete[] upright;
}

void annulus::glue_squares()
{
  for (int i = 0; i < length; i++)
  {
    int j;

    // Check for self-intersection
    for (j = i + 1; j < length; j++)
      if (sq[i] == sq[j])
      {
        cerr << "Error: an annulus is not allowed to intersect itself." << endl;
        exit(1);
      }

    // glue i to j:
    j = (i + 1)%length;

    // Depending on the orientations of i and j, glue some
    // vertical face of cube i to some vertical face of cube j.

    int toface = (upright[i] ? 1 : 0);
    perm howto = (upright[i] ? perm0213 : perm2103);

    (upright[i] ? sq[i]->lowright : sq[i]->midleft)->gluesym(
      (upright[j] ? sq[j]->lowleft : sq[j]->lowright),
      toface, howto);

    (upright[i] ? sq[i]->topright : sq[i]->topleft)->gluesym(
      (upright[j] ? sq[j]->topleft : sq[j]->midright),
      toface, howto);
  }
}

// Constructs a 2-handle, and glues it
// above or below this annulus.
annulus &annulus::twohandle(Direction is_above)
{
  int i;

  tetra *far, *mid, *near;
  tetra *farprev, *midprev, *nearprev;

  for(i = 0; i < length; i++)
  {
    // Glue together a bunch of prisms and
    // attach them to the tops of the squares.

    // construct three tetras
    far = new tetra(sq[i]->home, other);
    mid = new tetra(sq[i]->home, other);
    near = new tetra(sq[i]->home, other);

    // Build the prism

    // XOR since up/down inverts uprightness
    if(upright[i] ^ is_above)
      far->gluesym(mid, 3, perm0321);
    else
      far->gluesym(mid, 3, perm0132);

    mid->gluesym(near, 3, perm3120);
    

    // Glue prism to the floor / ceiling
    if (is_above)
    {
      far->gluesym(sq[i]->topleft, 0, perm1023);

      if (upright[i])
        mid->gluesym(sq[i]->topright, 0, perm2103);
      else
        mid->gluesym(sq[i]->topright, 0, perm(2,0,3,1));
    }
    else
    {
      far->gluesym(sq[i]->lowleft, 0, perm(3,2,0,1));

      if (upright[i])
        mid->gluesym(sq[i]->lowright, 0, perm(3,0,1,2));
      else
        mid->gluesym(sq[i]->lowright, 0, perm(3,2,0,1));
    }

  }

  // The next bit of code depends on the values of
  // "far", "mid", and "near" left over from the
  // end of the previous 'for' loop.
  
  for (i = 0; i < length; i++)
  {
    int prev = (i + length - 1)%length;

    // Play pass the pointer.
    farprev = far;
    midprev = mid;
    nearprev = near;

    far = is_above?
      sq[i]->topleft->thruface(1):
      sq[i]->lowleft->thruface(3);
    mid = far->thruface(3);
    near = mid->thruface(3);

    int toface = is_above? 2 : 1;
  
    // Glue the prisms to each other.
    near->gluesym(nearprev, toface, perm0213);

    (upright[i] ? far : mid)->gluesym(
      (upright[prev] ? midprev : farprev),
      toface, perm0213);
  }

  return *this;
}

annulus &annulus::operator++()
{
  twist(plus);
  return *this;
}

annulus &annulus::operator--()
{
  twist(minus);
  return *this;
}

// This inserts a "speed bump", i.e. solid 
// torus, above the surface cross interval.
void annulus::twist(Sign whichway)
{
  // There's a much worse way to do this, which for
  // obscenely large values of "length" (>6?)
  // actually uses fewer tetrahedra.
  // (The idea is: retriangulate the annulus so that 
  // it has length 1 and then do the twist(s).  Then
  // undo the retriangulation.  But this is pointless, 
  // SnapPea will totally change the triangulation, 
  // anyway.   A more general notion is to perform a long 
  // twist by conjugating it to a short one...)

  // Every speed bump consists of "length" identical layers.
  // Don't peel them apart.  
  // Never look inside a speed bump.  Trust us.

  tetra *tet[length][length];
  int i, j;

  // Create a bunch of tetrahedra. We'll refer
  // to tet[i][j] as the new tetrahedron in the
  // i^th row (from the bottom) and j^th column
  // (from the left). 
  
  for( i = 0; i < length; i++ )
    for( j = 0; j < length; j++ )
      // create a new tetrahedron and point tet[i][j] at it;
      tet[i][j] = new tetra(sq[j]->home, other);

  // First thing to do is glue all tetras in a column
  // along their "blue faces".  (For terminology, refer
  // to a certain paper tablecloth in the new Thai
  // restaurant on Bancroft above Telegraph.)
  // Thus loop over j first (sorry 'bout that). 
  // While we're at it we'll attach the last 
  // and first blue face to the ceiling and floor.
  //
  // Keep gluing 'til you're blue in the face.

  for( j = 0; j < length; j++ ) 
  {
    for( i = 0; i < length - 1; i++ )
      tet[i][j]->gluesym(tet[i+1][j], 2, perm0132);

    // Attach the top of the column to the ceiling.
    if( upright[j] )
      sq[j]->topright->subbedby(tet[length - 1][j], 2, perm0123); 
    else
      sq[j]->topright->subbedby(tet[length - 1][j], 2, perm(1,3,2,0));

    // Attach the bottom of the column to the floor.
    if( upright[j] )
      sq[j]->topright->gluesym(tet[0][j], 2, perm0132); 
    else
      sq[j]->topright->gluesym(tet[0][j], 2, perm(1,2,3,0));
  }

  // That finishes off the blue faces.  
  // Now to glue neighboring columns together
  // along the "red" faces. While we're at it
  // we'll attach the left-over red faces to the 
  // ceiling and floor. This is a bit tricky, so 
  // watch carefully. Follow the red card...

  for( j = 0; j < length; j++ )
  {
    int next = (j + 1)%length;
    int begin, end;

    // Assuming here that true == 1 and false == 0.
    int offset = whichway*(!upright[j] + upright[next]);

    if(whichway == plus)
    {
      begin = offset;
      end = length;
    }
    else
    {
      begin = 0;
      end = length + offset;
    }

    // Gluing the columns together.
    for( i = begin; i < end; i++ )
    {
      tet[i][j]->gluesym(tet[i - offset][next], 1, perm1023);
    }

    // Now glue the left over red faces on top to the 
    // ceiling using subbedby. Then glue remaining red
    // faces (on the bottom) to the newly available floor.

    if( upright[j] && upright[next] )
    {
      // The offset is whichway*1.
      if(whichway == plus)
      {
        sq[next]->topleft->subbedby(tet[length - 1][next], 1, perm(1,0,3,2)); 
        sq[next]->topleft->gluesym(tet[0][j], 1, perm0132); 
      }
      else
      {
        sq[next]->topleft->subbedby(tet[length - 1][j], 1, perm0123); 
        sq[next]->topleft->gluesym(tet[0][next], 1, perm1023); 
      }
    }      
    if( upright[j] && !upright[next] )
    {
      // The offset is zero, so there are 
      // no left-over red faces.  :)
    }      
    
    if( !upright[j] && upright[next] )
    {
      // The offset is whichway*2.
      if(whichway == plus)
      {
        sq[next]->topleft->subbedby(tet[length - 1][next], 1, perm(1, 0, 3, 2)); 
        sq[next]->topleft->gluesym(tet[1][j], 1, perm0132); 

        sq[j]->topleft->subbedby(tet[length - 2][next], 1, perm(2, 0, 1, 3)); 
        sq[j]->topleft->gluesym(tet[0][j], 1, perm2103); 
      }
      else
      {
        sq[j]->topleft->subbedby(tet[length - 1][j], 1, perm(3, 1, 0, 2)); 
        sq[j]->topleft->gluesym(tet[1][next], 1, perm(3, 0, 1, 2)); 

        sq[next]->topleft->subbedby(tet[length - 2][j], 1, perm0123); 
        sq[next]->topleft->gluesym(tet[0][next], 1, perm1023); 
      }
    }      
    if( !upright[j] && !upright[next] )
    {
      // The offset is whichway*1.
      if(whichway == plus)
      {
        sq[j]->topleft->subbedby(tet[length - 1][next], 1, perm(2, 0, 1, 3)); 
        sq[j]->topleft->gluesym(tet[0][j], 1, perm2103); 
      }
      else
      {
        sq[j]->topleft->subbedby(tet[length - 1][j], 1, perm(3, 1, 0, 2)); 
        sq[j]->topleft->gluesym(tet[0][next], 1, perm(3, 0, 1, 2)); 
      }
    }      
    // Isn't that all perfectly clear?
    // No?  Well, is my face is red!
  }
}    

///////////// Rectangle code ///////////////////////

rectangle::rectangle(orisquare &ori)
{
  length = ori.depth;

  orisquare *iter=&ori, *next;

  sq = new square*[length];
  upright = new bool[length];

  for (int i=0; i<length; i++)
  {
    sq[i] = iter->sq;
    upright[i] = iter->is_upright;
    next = iter->next;
    delete iter;
    iter = next;
  }

  glue_squares();
}

rectangle::rectangle(int length_in, square **sq_in, bool *upright_in)
{
  length = length_in;

  sq = new square*[length];
  upright = new bool[length];

  for (int i=0; i<length; i++)
  {
    sq[i] = sq_in[i];
    upright[i] = upright_in[i];
  }

  glue_squares();
}

rectangle::~rectangle()
{
  delete[] sq;
  delete[] upright;
}

// same as the code for annuli, except we don't glue cyclically
void rectangle::glue_squares()
{
  for (int i=0; i<length-1; i++)
  {
    // glue i to j:
    int j=i+1;

    // Depending on the orientations of i and j, glue some
    // vertical face of cube i to some vertical face of cube j.

    int toface = upright[i] ? 1 : 0;
    perm howto = upright[i] ? perm0213 : perm2103;

    (upright[i] ? sq[i]->lowright : sq[i]->midleft)->gluesym(
      (upright[j] ? sq[j]->lowleft : sq[j]->lowright),
      toface, howto);

    (upright[i] ? sq[i]->topright : sq[i]->topleft)->gluesym(
      (upright[j] ? sq[j]->topleft : sq[j]->midright),
      toface, howto);
  }
}
///////////// Manifold code //////////////////

manifold::manifold(char *name_in)
{
  name = name_in;
  cusp_count = 0;

  last = NULL;
}

manifold::~manifold()
{
  // Delete all tetrahedra 
  // by traversing the list.
  tetra *behind;

  while(last)
  {
    behind = last;
    last = last->lookback();
    delete behind;
  }
  return;
}

tetra *manifold::onemore(tetra *newguy)
{
  tetra *temp = last;
  last = newguy;
  return temp;
}

void manifold::capoff()
{
  if (!last)  // empty manifold?
    return;

  tetra *current = last;

  int firstcount = size(); // Actually don't need this;
                           // but probably faster this way.
  while (current)
  {
    current->grow(this);
    current = current->lookback();
  }

  current = last;  // last of the NEW tetras

  while (current->number() >= firstcount) // (Just check 'current != NULL'.)
  {
    current->neighborglue();
    current = current->lookback();
  }
}

void manifold::bundle_capoff()
{
  int onecusp, twocusp, oldcusp, newcusp;
  if (!last)  // empty manifold?
    return;

  tetra *current = last;
  tetra *incurrent;

  int firstcount = size(); // Actually don't need this;
                           // but probably faster this way.
  while (current)
  {
    current->bundle_grow(this);
    current = current->lookback();
  }

  // Round 1
  current = last;  // last of the NEW tetras

  while (current->number() >= firstcount) // (Just check 'current != NULL'.)
  {
    current->ups_to_downs();
    current = current->lookback();
  }

  // Round 2
  current = last;  // last of the NEW tetras

  while (current->number() >= firstcount) // (Just check 'current != NULL'.)
  {
    int fromface;
    onecusp = current->whatcusp();
    for (fromface = 0; fromface < 3; fromface++)
    {
      if(debugging)
	{ 
	  cerr << "round2_" << current->number() << "_" << fromface << " " << flush;
	}
      twocusp = current->bundle_neighborglue(fromface);
      if (onecusp == twocusp)
        continue;
      newcusp = (onecusp > twocusp)? onecusp: twocusp;
      oldcusp = (onecusp < twocusp)? onecusp: twocusp;

      incurrent = last;
      while (incurrent->number() >= firstcount)
      {
        if (incurrent->whatcusp() == oldcusp)
        {
          incurrent->setcusp(newcusp);
          incurrent->you_lose();
        }
        incurrent = incurrent->lookback();
      }
    }

    current = current->lookback();
  }

  current = last;  // last of the NEW tetras
  while (current->number() >= firstcount) // (Just check 'current != NULL'.)
  {
    if ((oldcusp = current->whatcusp()) > firstcount)
    {
      incurrent = last;
      while (incurrent->number() >= firstcount)
      {
        if (incurrent->whatcusp() == oldcusp)
        {
          incurrent->setcusp(cusp_count);
        }
        incurrent = incurrent->lookback();
      }
      cusp_count++;
    }
    current = current->lookback();
  }
}

void tetra::ups_to_downs()
{
  int fromface = gluing[3][3];
  // Error check
  if (gluedto[fromface] != NULL)
  {
    cerr << "Incorrect assumption in ups_to_downs" << endl;
    exit(1);
  }
  if (position == up)
  {
    if(debugging)
      { 
	cerr << "updownwalk " << flush;
      }
    walk_about(fromface);
    // No one will ever see this.
    if (gluedto[fromface]->position != down)
    {
       cerr << "Tear your hair out." << endl;
       exit(1);
    }
    cusp = index;
    gluedto[fromface]->cusp = cusp;
  }
}

int tetra::bundle_neighborglue(int fromface)
{
  if (gluedto[fromface] == NULL)
  {
    if(debugging)
      { 
	cerr << "bunwalk " << flush;
      }
    walk_about(fromface);
  }
  else
  {
    // This is just error checking.
    if (gluedto[fromface]->cusp != cusp)
    {
      cerr << "Some cusp wasn't properly guillotined." << endl;
      exit(1);
    }
  }
  return gluedto[fromface]->cusp;
}

///////////// For printing //////////////////

ostream &operator<<(ostream &o, const perm &to_print)
{
  for (int i=0; i<4; i++)
  {
    o << to_print.image[i];
  }
  return o;
}

void manifold::snap_print(ostream &o)
{
  int i, ref;

  tetra *t;

  if (last)
    ref = last->number();

  o << "% Triangulation" << endl;
  o << (char *) name << endl;
  o << "not_attempted  0.00000000" << endl;

  //  We used to print
  // o << "unknown_orientability" << endl;
  // here.  However, all manifolds we build are certain to to be
  // orientable, as all face gluings are odd.
  o << "oriented_manifold" << endl;

  o << "CS_unknown" << endl;
  o << endl;

  // Print the number of torus cusps (cusp_count) and the number of
  // Klein bottle cusps (0).
  o << cusp_count <<" 0" << endl;

  // If there are any cusps, well, we don't know their shapes.  So
  // print zeros.
  for(i=0; i<cusp_count; i++)
    o << "    torus   0.000000000000   0.000000000000" << endl;
  o << endl;

  // Print the number of tetrahedra.
  o << size() << endl;

  // Print the face gluings for each tetrahedron and the cusp data, if
  // any.
  t = last;
  while (t)
  {
    t->print_wrt(o, ref);
    t = t->lookback();
  }

  o << endl;
}

void tetra::print_wrt(ostream &o, int lastnumber)
{
  int fromface = gluing[3][3];

  // This is how many times the meridian (if it exists) and longitude
  // cross the sides of the link of vertex three of tetrahedron t.
  // const static char *m_up[3] = {" 1 -1  0  0", " 0  1 -1  0", " 0 -1  1  0"};
  // const static char *m_down[3] = {" 1  0 -1  0", "-1  1  0  0", "-1  0  1  0"};

  // Corrected orientation?
  const static char *m_up[3] = {"-1  1  0  0", " 0 -1  1  0", " 0  1 -1  0"};
  const static char *m_down[3] = {"-1  0  1  0", " 1 -1  0  0", " 1  0 -1  0"};

  const static char *l_up[3] = {"-1  0  1  0", " 1 -1  0  0", "-1  0  1  0"};
  const static char *l_down[3] = {" 0 -1  1  0", " 1  0 -1  0", "-1  1  0  0"};

  const static char *filler = " 0  0  0  0";

  for (int i=0; i<4; i++)
  {
    if (!gluedto[i]) // Don't give up if there is no gluing info.
    {
      o << "   ?";
      cerr << "Free face encountered: producing invalid SnapPea file" << endl;
    }
    else
      o << "   " << lastnumber - gluedto[i]->number();
  }
  o << endl;

  for (int i=0; i<4; i++)
  {
    if (!gluedto[i]) // Don't give up if there is no gluing info.
      o << " ????";
    else
      o << " " << gluing[i];
  }
  o << endl;

  // Material vertices 0 through 2
  for (int i=0; i<3; i++)
    o << "  -1 ";
  o << "  " << cusp << " " << endl;

  //  meridian  (right sheet)
  o << "  0  0  0  0  0  0  0  0  0  0  0  0 ";
  if (meridian && cusp > -1)
  {
    if (position == up)
    {
      o << m_up[fromface] << endl;
    }
    else
    {
      o << m_down[fromface] << endl;
    }
  }
  else
  {
    o << filler << endl;
  }

  //  meridian  (left  sheet)
  o << "  0  0  0  0  0  0  0  0  0  0  0  0  0  0  0  0" << endl;

  //  longitude (right sheet)
  o << "  0  0  0  0  0  0  0  0  0  0  0  0 ";
  if (cusp > -1)
  {
    if (position == up)
    {
      o << l_up[fromface] << endl;
    }
    else
    {
      o << l_down[fromface] << endl;
    }
  }
  else
  {
    o << filler << endl;
  }

  //  longitude (left  sheet)
  o << "  0  0  0  0  0  0  0  0  0  0  0  0  0  0  0  0" << endl;

  o << "  0.000000000000   0.000000000000" << endl;

  o << endl;
}
