#include <iostream.h>
#include <string.h>

#include "twister.h"  

int main(int argc, char **argv)
{
  // A manifold starts out with only a name.  Although it is
  // possible to have more than one manifold in memory,
  // typically there is only one.

  char *name;

  if (argc == 2)
    name = argv[1];    // Name the bundle for its monodromy
  else
    name = "Trivial_bundle";

  manifold M(name);  

  // Now to build a surface.  This is done using some
  // collection of closed curves and arcs which "fill" the
  // surface--that is, the complement of the curves should
  // be a collection of topological disks, each of which
  // touches the set of punctures no more than once.

  // The collection of curves should also include any
  // curve along which we wish to perform Dehn twists,
  // and any curves to which we wish to attach 2-handles.
  // No curve may cross itself, but a given pairs of curves
  // may cross each other multiple times.

  // In this example, we construct a once-punctured surface of
  // genus two, with a standard set of closed curves (named
  // $a$, $b$, $c$, $d$, and $e$) and an arc $x$ which has both
  // endpoints in the single puncture.  The curves $b$ and $d$
  // go around the "holes" on the left and right, respectively;
  // $a$ and $e$ cut through the left and right "handles" of
  // the surface, and $c$ cuts between the holes.  The arc $x$
  // crosses through $d$ once before returning to the puncture.

  // Here's the intersection graph of the curves:
  //
  //    a   e
  //    |   |
  //    b-c-d-x
  //
  // which is actually all the information we need, in this
  // simple case, to construct the surface.

  // The curves are in fact oriented, although the choice of
  // orientation has no effect on the manifold constructed.

  // Every point of intersection between curves becomes a
  // square of the squared surface.  There are five of them in
  // this example, one for each edge of the intersection graph.

  // A square needs to know which manifold it belongs to, so
  // the constructor takes one argument.  We could name the
  // squares individually, like
  //
  //   square ab(M), bc(M), cd(M), de(M), dx(M);
  // 
  // or keep them in an array: the individual squares will be
  // named s[0], s[1], s[2], s[3], and s[4].

  square s[5] = {  square(M), square(M), square(M),
                              square(M), square(M)  };

  // Each closed curve in the surface becomes an annulus in
  // the squared surface, and each arc becomes a rectangle.
  // An annulus or rectangle is an ordered list of oriented
  // squares, according to the following convention:  follow
  // the curve along the surface in the direction of its own
  // orientation, and give signs to the intersections according
  // to the orientation of the curve being crossed: 
  //
  //      |            |           |
  //      ^            |           |
  //      | 1st        | 2nd       | 3rd
  //    --+-->---------+-->--------+-->-------- etc.
  //      |'plus'      |'minus'    |'minus'
  //      |            v           v
  //      |            |           |
  //  
  // After all of the curves have been specified, each square
  // will be listed exactly twice, once with 'plus' orientation
  // and once with 'minus'.  For an example as simple as this
  // one, we can make arbitrary choices for the orientations
  // of the squares, and still be assured that our convention
  // arises from some choice of orientations on the curves.
  //
  // We refer once more to the intersection graph.  This time
  // each edge has been labelled to indicate which square it
  // corresponds to, s[0] through s[4].
  //
  //    a       e
  //    |       |
  //    |0      |3
  //    |       |
  //    b---c---d---x
  //      1   2   4
  //
  // We are ready to construct the annuli and rectangle:

  annulus a(+s[0]);
  annulus b(-s[0] >> +s[1]);
  annulus c(-s[1] >> +s[2]);
  annulus d(-s[2] >> +s[3] >> +s[4]);
  annulus e(-s[3]);

  rectangle x(-s[4]);

  // The argument to an annulus or rectangle constructor is a
  // linked list of oriented squares.  An object of class
  // 'orisquare' (oriented square) is produced by applying
  // either the unary '+' operator or the unary '-' operator
  // to an object of class 'square', and the '>>' operator
  // has been overloaded to concatenate oriented squares into
  // a linked list.  A single oriented square is already a
  // linked list of length one.
  // 
  // (The unary '+' and '-' operators allocate memory for the
  // oriented square they create, which is then deleted by
  // the annulus and rectangle constructors.  Since the only
  // reason to create an oriented square is to form part of
  // an annulus or rectangle, and each oriented square can only
  // be used once, this should cause no problems.)

  // The manifold $M$ is already the cartesian product of our surface
  // with the interval, and from this point there are several things
  // we could do.  To construct a bundle over the circle, we need to
  // glue the surface cross {0} to the surface cross {1}.  This is
  // done by telling each square (which has actually been thickened to
  // a cube) to glue its top face to its bottom face:

  for (int i=0; i<5; i++)
    s[i].closeup();

  // If instead we wished to create a Heegard splitting, we would not
  // close up the squares, but rather attach 2-handles to the upper
  // and lower surfaces, along annuli:

  //   a.twohandle(below);
  //   e.twohandle(below);
  //
  //   b.twohandle(above);
  //   d.twohandle(above);
  //
  // (We could also glue in fewer handles to create a compression
  // body.)  Naturally, every closed curve along which we attach a
  // 2-handle above must be disjoint from other 2-handles above, and
  // similarly for 2-handles glued below. Handles may only be glued to
  // annuli, not to rectangles.
  //
  // Before we do anything further, we need to get rid of free faces
  // of tetrahedra coming from the boundary of $M$.  Following the
  // conventions of SnapPea, every 2-sphere component of the boundary
  // of $M$ will be filled in with a ball, and boundary components
  // of genus one or greater will be represented as the (non-sphere)
  // link of a vertex "at infinity".  A single function accomplishes
  // this:

  M.capoff();

  // Now to the heart of the twister.c package: we can unglue our
  // manifold along the surface, and reglue after having performed
  // a Dehn twist (positive or negative) along any of the named
  // annuli.  Composition of Dehn twists allows us to construct any
  // monodromy (for surface bundles) or Heegard splitting, provided
  // the set of curves we specified was rich enough to generate the
  // mapping class group.  (The mapping class group which fixes
  // punctures individually, that is--we are limited to full twists,
  // not half twists.)  The conventions have been chosen to allow
  // reading the word from left to right, so we think of Dehn twists
  // as acting on the right.  (Not that it makes any difference.)
  //
  // This routine reads the first argument after the command name and
  // interprets it as a word in positive and negative Dehn twists:

  if (argc == 2)
    for( int i = 0; argv[1][i] != '\000'; i++ )
    {
      if( argv[1][i] == 'a' )
        a.twist(plus);
      if( argv[1][i] == 'A' )
        a.twist(minus);

      if( argv[1][i] == 'b' )
        b.twist(plus);
      if( argv[1][i] == 'B' )
        b.twist(minus);

      if( argv[1][i] == 'c' )
        c.twist(plus);
      if( argv[1][i] == 'C' )
        c.twist(minus);

      if( argv[1][i] == 'd' )
        d.twist(plus);
      if( argv[1][i] == 'D' )
        d.twist(minus);

      if( argv[1][i] == 'e' )
        e.twist(plus);
      if( argv[1][i] == 'E' )
        e.twist(minus);
    }

  // Finally, we print out the resulting triangulated manifold in a
  // format understood by SnapPea.  All we do is glue the thing
  // together; the hard work we leave to SnapPea.
  
  M.snap_print(cout);
}
